with Reps.Ops, Round_Up, Reps.IO,Messages;
with Ada.Text_IO;
use Ada.Text_IO, Reps.IO,Messages;

pragma Elaborate_All (Reps.Ops);

package body Intervals.Ops is

   NegOne:  constant Rep := -One;
   NegTwo:  constant Rep := -Two;
   NegHalf: constant Rep := -Half;
   Three:   constant Rep := Rep(3);
   Quarter: constant Rep := Half/Two;
   Epsilon: constant Rep := Half**52; --was 99

   Interval_Zero: constant Interval := (Zero,Zero);
   Interval_One:  constant Interval := Scal(One);
   PlusMinus:     constant Interval := (Zero,One);


   function MinusPi_Pi(S: Interval) return Interval is
      Interval_2Pi: constant Interval:=Two*Scal(Rep(3.14159_26535_89793_23),Rep(3.14159_26535_89793_24));
      Interval_Pi: constant Interval:=Scal(Rep(3.14159_26535_89793_23),Rep(3.14159_26535_89793_24));
      SS: Interval:=S;
   begin
      while Inf(SS)>=Sup(Interval_2Pi) loop
         SS:=SS-Interval_2Pi;
      end loop;
      while Sup(SS)<=Zero loop
         SS:=SS+Interval_2Pi;
      end loop;
      if Inf(SS)>Sup(Interval_Pi) then
         SS:=SS-Interval_2Pi;
      end if;
      return SS;
   end MinusPi_Pi;

   function Contains(R1,R2: Interval) return Boolean is
   begin
      return  Inf(R1) <= Inf(R2) and Sup(R1) >= Sup(R2);
   end Contains;

   function Scal(R1,R2: Rep) return Interval is
      Sum: constant Rep := R1+R2;
   begin
      if R1 >= R2 then
         return (Half*Sum,Half*((R1-R2)+(Sum+(-R1-R2))));
      end if;
      return (Half*Sum,Half*((R2-R1)+(Sum+(-R1-R2))));
   end Scal;

   function Ball(S: Interval) return Interval is
   begin
      return (Zero,Abs(S.C)+S.R);
   end Ball;

   function Inf(S: Interval) return Rep is
   begin
      return -(S.R-S.C);
   end Inf;

   function Sup(S: Interval) return Rep is
   begin
      return S.C+S.R;
   end Sup;

   pragma Inline (Inf,Sup);

   function SupAbs(S: Interval) return Rep is
   begin
      return Abs(S.C)+S.R;
   end SupAbs;

   function "<"(S1,S2: Interval) return Boolean is
   begin
      if Sup(S1)<Inf(S2) then return True; end if;
      if Sup(S2)<=Inf(S1) then return False; end if;
      raise Undefined;
      return False;
   end "<";

   function IntFloor(S: Interval) return Integer is
      R: constant Rep := Rep'Floor(Inf(S));
   begin
      if Rep'Floor(Sup(S)) /= R then
         raise Undefined;
      end if;
      return Integer(R);
   end IntFloor;

   function IntCeiling(S: Interval) return Integer is
      R: constant Rep := Rep'Ceiling(Sup(S));
   begin
      if Rep'Ceiling(Inf(S)) /= R then
         raise Undefined;
      end if;
      return Integer(R);
   end IntCeiling;

   function "abs"(S: Interval) return Interval is
      R: Rep;
   begin
      if S.C >= S.R then
         return S;
      elsif -S.C >= S.R then
         return (-S.C,S.R);
      end if;
      R := Half*(Abs(S.C)+S.R);
      return (R,R);
   end "abs";

   function Min(S1,S2: Interval) return Interval is
      U1: constant Rep := Sup(S1);
      L2: constant Rep := Inf(S2);
      L1,U2: Rep;
   begin
      if U1 <= L2 then
         return S1;
      end if;
      L1 := Inf(S1);
      U2 := Sup(S2);
      if U2 <= L1 then
         return S2;
      end if;
      return Scal(Rep'Min(L1,L2),Rep'Min(U1,U2));
   end Min;

   function Max(S1,S2: Interval) return Interval is
      U1: constant Rep := Sup(S1);
      L2: constant Rep := Inf(S2);
      L1,U2: Rep;
   begin
      if U1 <= L2 then
         return S2;
      end if;
      L1 := Inf(S1);
      U2 := Sup(S2);
      if U2 <= L1 then
         return S1;
      end if;
      return Scal(Rep'Max(L1,L2),Rep'Max(U1,U2));
   end Max;

   function Cap(R: Radius; S: Interval) return Interval is
      -- bound on characteristic_function for closed ball of radius R
   begin
      if Abs(S.C)+S.R <= R then
         return S;
      end if;
      return Scal(Rep'Max(-R,Rep'Min(Zero,Inf(S))),Rep'Min(R,Rep'Max(Zero,Sup(S))));
   end Cap;

   function Up(R: Rep; Dummy: Interval) return Rep is
      -- only used in high level procedures to increase domains
   begin
      return R+Epsilon;         -- avoid Gnat's Succ in Round_Up mode
   end Up;

   function Widen(S: Interval) return Interval is
      -- only used in high level procedures to increase norms
   begin
      return (S.C,S.R+Epsilon); -- avoid Gnat's Succ in Round_Up mode
   end Widen;

   procedure ErrMult(R: in Rep; S: in out Interval) is
   begin
      S.R := Abs(R)*S.R;
   end ErrMult;

   function "+"(S,T: Interval) return Interval is
      Sum: constant Rep := S.C+T.C;
   begin
      return (Sum,S.R+T.R+(Sum+(-S.C-T.C)));
   end "+";

   function "-"(S,T: Interval) return Interval is
      Diff: constant Rep := S.C-T.C;
   begin
      return (Diff,S.R+T.R+(Diff+(T.C-S.C)));
   end "-";

   function "*"(R: Rep; S: Interval) return Interval is
      Prod: constant Rep := R*S.C;
   begin
      return (Prod,abs(R)*S.R+(Prod+(-R)*S.C));
   end "*";

   function "*"(S: Interval; R: Rep) return Interval is
      Prod: constant Rep := R*S.C;
   begin
      return (Prod,Abs(R)*S.R+(Prod+(-R)*S.C));
   end "*";

--   function "/"(S: Interval; R: Rep) return Interval is
--      Quot: constant Rep := S.C/R;
--   begin
--      return (Quot,S.R/Abs(R)+(Quot+S.C/(-R)));
--   end "/";

   function "/"(S: Interval; R: Rep) return Interval is
      Rt: Rep;
   begin
      Rt := S.C/R;
      return(Rt,Rt*Epsilon+S.R/R);
   end "/";

   function Sqr(S: Interval) return Interval is
      Prod:  Rep := S.C*S.C;
   begin
      if S=Interval_Zero or S=Interval_One  then
         return S;
      end if;
     -- Put("S.C=");Txt_Put(Current_Output,S.C);
      Prod:= S.C*S.C;
     -- Put("here");
      --New_Line;
      --Put("Prod=");Txt_Put(Current_Output,Prod);
      return (Prod,(Two*Abs(S.C)+S.R)*S.R+(Prod+S.C*(-S.C)));
   end Sqr;

   function "*"(S,T: Interval) return Interval is
      Prod: constant Rep := S.C*T.C;
   begin
      if S=Interval_Zero or T=Interval_Zero then
         return Interval_Zero;
      end if;
      return (Prod,Abs(S.C)*T.R+S.R*(Abs(T.C)+T.R)+(Prod+S.C*(-T.C)));
   end "*";


--   function "*"(S,T: Interval) return Interval is
--      Prod: constant Rep := S.C*T.C;
--   begin
--      if S=Interval_Zero or T=Interval_Zero then
--         return Interval_Zero;
--      end if;
--      return (Prod,Prod*Epsilon+S.C*T.R+T.C*S.R+S.R*T.R);
--   end "*";



--   function Inv(S: Interval) return Interval is
--      Quot: constant Rep := One/S.C;
--   begin
--      if Abs(S.C) <= S.R then
--         raise Constraint_Error;               -- division by zero
--      end if;
--      return (Quot,S.R/Abs(S.C)/(-(S.R-Abs(S.C)))+(Quot+NegOne/S.C));
--   end Inv;

   function Inverse(S: Interval) return Interval is
      R0,R1,R2: Rep;
   begin
      R1 := Rep'Pred(Half/Sup(S));
      R2 := Rep'Succ(Half/Inf(S));
      if R1<Zero or else R2>Zero then
         R0 := R2+R1;
         R1 := Rep'Succ(R2-R1);
      else
         Message("Interval_Ops.Inv",Division_By_Zero);
      end if;
      return (R0,R1+R0*Epsilon);
   end Inverse;

   function Inv(S: Interval) return Interval is
      C: Interval;
      E: constant Interval:=Scal(-Err(S),Err(S));
   begin
    C:=Inverse(Scal(Approx(S)));
    if Err(S)/=Zero then
       C:=C-E*Sqr(Inverse(S));
    end if;
    return C;
   end Inv;

   function "/"(S,T: Interval) return Interval is
   begin
      if S=Interval_Zero  then
         return Interval_Zero;
      end if;
      return S*Inv(T);
   end "/";

   function "/"(R: Rep; S: Interval) return Interval is
   begin
      return R*Inv(S);
   end "/";

   function "**"(S: Interval; I: Integer) return Interval is
   E: constant Interval:=Scal(-Err(S),Err(S));
   begin
      if S=Interval_One then
         return Interval_One;
      end if;
      if S=Interval_Zero and I>=0 then
         return Interval_Zero;
      end if;
      if S=Interval_Zero and I<0 then
          Message("**.Intervals-Ops",Division_By_Zero);
      end if;
      if I>0 then
         declare
            J: Integer := I;
            X,Y,Xa,Ya: Interval;
         begin
            X := S;
            Xa:=Scal(Approx(S));
            if (J rem 2)=0 then
               Y := Interval_One;
               Ya := Interval_One;
            else
               Y := X;
               Ya:=Xa;
            end if;
            J := J/2;
            while J>0 loop
               X := Sqr(X);
               Xa:=Sqr(Xa);
               if (J rem 2)>0 then
                  Y := X*Y;
                  Ya:=Xa*Ya;
               end if;
               J := J/2;
            end loop;
--           return Ya*Scal(Approx(S))+Rep(I)*E*Y;
            return Y;
         end;
      elsif I<0 then
         return Inv(S)**(-I);
      else
         return Interval_One;
      end if;
   end "**";

   procedure CoshSinh(S: in Interval; Sc,Ss: out Interval) is
      Iter: constant Integer := 8;
   begin
      if S = Interval_Zero then
         Sc := Interval_One;
         Ss := S;
      else
         declare
            K: Integer;
            R: Rep;
            X,T: Interval;
         begin
            K := 0;
            R := SupAbs(S);
            while R>Half loop
               K := K+1;
               R := R*Half;
            end loop;
            X := (Half**K)*S;
            T := Sqr(X);
            Sc := (One,Half);
            Ss := Sc;
            for I in reverse 1 .. Iter loop
               Sc := Interval_One+Sc*T/Rep((2*I)*(2*I-1));
               Ss := Interval_One+Ss*T/Rep((2*I)*(2*I+1));
            end loop;
            Ss := X*Ss;
            for I in 1 .. K loop
               T := Sc;
               Sc := Interval_One+Two*Sqr(Ss);
               Ss := Two*(T*Ss);
            end loop;
         end;
      end if;
   end CoshSinh;


   procedure CosSin(S: in Interval; Sc,Ss: out Interval) is
      Iter: constant Integer := 8;
      Interval_Pi: constant Interval:=Scal(Rep(3.14159_26535_89793_23),Rep(3.14159_26535_89793_24));
      Co: Interval;
   begin
      if S = Interval_Zero then
         Sc := Interval_One;
         Ss := S;
      else
         declare
            K: Integer;
            R: Rep;
            X,T: Interval;
         begin
            K := 0;
            R := SupAbs(S);
            while R>Half loop
               K := K+1;
               R := R*Half;
            end loop;
            X := (Half**K)*S;
            T := Sqr(X);
            Sc := (One,Half);
            Ss := Sc;
            if ((Iter-1) mod 2)=0 then
               Co:=Interval_One;
            else
               Co:=-Interval_One;
            end if;
            for I in reverse 1 .. Iter loop
               Sc := Co+Sc*T/Rep((2*I)*(2*I-1));
               Ss := Co+Ss*T/Rep((2*I)*(2*I+1));
               Co:=-Co;
            end loop;

            Ss := X*Ss;
            for I in 1 .. K loop
               T := Sc;
               Sc := Interval_One-Two*Sqr(Ss);
               Ss := Two*(T*Ss);
            end loop;
         end;
      end if;
   end CosSin;


   function Cosh(S: Interval) return Interval is
      Sc,Ss: Interval;
   begin
      CoshSinh(S,Sc,Ss);
      return Sc;
   end Cosh;

   function Sinh(S: Interval) return Interval is
      Sc,Ss: Interval;
   begin
      CoshSinh(S,Sc,Ss);
      return Ss;
   end Sinh;

   function Sin(S: Interval) return Interval is
      Sc,Ss: Interval;
   begin
      CosSin(S,Sc,Ss);
      return Ss;
   end Sin;

   function Cos(S: Interval) return Interval is
      Sc,Ss: Interval;
   begin
      CosSin(S,Sc,Ss);
      return Sc;
   end Cos;


   function Exp(S: Interval) return Interval is
   begin
      if S=Interval_Zero then return Scal(One); end if;   -- S=0
      if Sup(S)<Zero then
         return Inv(Exp(-S));
      end if;    -- S<0
      declare
         X,Y: Interval;
      begin
         if Inf(S)<Zero then                               -- S contains 0
            CoshSinh(S,X,Y);
            return X+Y;
         end if;
         declare                                           -- S>0
            Iter: constant Integer := 128;
            K: Integer := 0;
            R: Rep;
         begin
            R := Sup(S);
            while R>Quarter loop
               K := K+1;
               R := R*Half;
            end loop;
            X := (Half**K)*S;
            Y := (Zero,Three/Two);
            for I in reverse 1 .. Iter loop
               Y := Interval_One+(X*Y)/Rep(I);
            end loop;
            for I in 1 .. K loop
               Y := Sqr(Y);
            end loop;
            return Y;
         end;
      end;
   end Exp;


   function Log(S: Interval) return Interval is
      use Reps.Ops,Reps.IO;
   begin
      if S=Interval_One then return Interval_Zero; end if;
      declare
         Iter: constant Integer := 128;
         S0,LogS0,X,Y: Interval;
      begin
         LogS0 := Scal(Log(Approx(S)));   --- guess
         S0 := Exp(LogS0);
         X := (S0-S)/S0;                  --- Log(S) = Log(S0)+Log(1-X)
         Y := Interval_One-Scal(SupAbs(X));
         if Inf(Y) <= Zero then
            New_Line;
            Message("Intervals.Ops.Log",Zero_In_Log);
         end if;
         Y := PlusMinus*Inv(Y)/Rep(Iter+1);
         for I in reverse 1 .. Iter loop
            Y := Inv(Scal(I))+X*Y;
         end loop;
         return LogS0-X*Y;
      end;
   end Log;


   function Sqrt(S: Interval) return Interval is
      S1,H,E: Interval;
      F: Rep;
   begin
      if Inf(S)<=Zero then
         New_Line;
         Message("Intervals.Ops.Sqrt",Zero_In_Log);
      elsif Sup(S)<Rep(1.0E-14) then
         S1:=Rep(1.0E+14)*S;
         F:=Rep(1.0E-7);
      elsif Sup(S)<Rep(1.0E-12) then
         S1:=Rep(1.0E+12)*S;
         F:=Rep(1.0E-6);
      elsif Sup(S)<Rep(1.0E-10) then
         S1:=Rep(1.0E+10)*S;
         F:=Rep(1.0E-5);
      elsif Sup(S)<Rep(1.0E-8) then
         S1:=Rep(1.0E+8)*S;
         F:=Rep(1.0E-4);
      elsif Sup(S)<Rep(1.0E-6) then
         S1:=Rep(1.0E+6)*S;
         F:=Rep(1.0E-3);
      elsif Sup(S)<Rep(1.0E-4) then
         S1:=Rep(1.0E+4)*S;
         F:=Rep(1.0E-2);
      elsif Sup(S)<Rep(1.0E-2) then
         S1:=Rep(1.0E+2)*S;
         F:=Rep(1.0E-1);
      else
         S1:=S;
         F:=One;
      end if;
      H:=Half*Log(S1);
      E:=Exp(H);
      return F*E;
   end Sqrt;


   function Lambda(M,N: Integer) return Interval is
      S: Interval;
   begin
      S := (Three/Two,Half);
      for I in 1 .. 100 loop
         S := Interval_One+Inv(S);
      end loop;
      return Scal(M)-Rep(N)*Inv(S);
   end Lambda;

   function Short_Exp(R: Interval; Iter: Integer) return Interval is
   begin
      return R;
   end Short_Exp;



begin

   Round_Up;



end Intervals.Ops;
